/** @file
  * Common system-specific functionality.  All functions are private - so
  * they require direct access.
  */

private import std.path;
private import std.c.stdio;
private import std.string;
private import std.c.windows.windows;
private import std.file;

private class sysFunctions
{
    import sysBase;
}

struct Stat
{
    char[] basename; /**< Filename without path. */
    char[] filename; /**< Filename with path. */
    char[] path; /**< The path of the filename. */
    char[] ext; /**< The filename's extension. */
    ulong size; /**< The size of the file in bytes. */
    bit isdir; /**< True if this is a directory, else this is a file. */
}

private char[] quotedString(char[] input)
{
    char[] result = "\"";

    for (int c; c < input.length; c ++)
    {
        char ch = input[c];

        if (ch == '\"') result ~= '\"';
        else if (ch == '\'') result ~= "\\'";
        else if (ch == '\n') result ~= '\n';
        else if (ch == '\a') result ~= '\a';
        else if (ch == '\?') result ~= '\?';
        else if (ch == '\\') result ~= '\\';
        else if (ch == '\b') result ~= '\b';
        else if (ch == '\f') result ~= '\f';
        else if (ch == '\r') result ~= '\r';
        else if (ch == '\t') result ~= '\t';
        else if (ch == '\v') result ~= '\v';
        else if (ch < 32 || (ch > 127 && ch <= 255))
        {
            char[3] o;

            sprintf(o, "%02X", cast (uint) ch);
            result ~= "\\x" ~ o[0..2];
        }
        else if (ch > 255)
        {
            char[5] o;

            sprintf(o, "%04X", cast (uint) ch);
            result ~= "\\u" ~ o[0..4];
        }
        else
            result ~= ch;
    }

    result ~= "\"";
    return result;
}

/** An invalid filename has been specified.  For example, it might be too
  * long, zero-length, or contain invalid characters.
  */

class InvalidFilenameError : Error
{
    char[] filename; /**< The filename which has caused this error. */

    this(char[] filename)
    {
        this.filename = filename;
        super(quotedString(filename) ~ " is an invalid filename.");
    }
}

private:

/** Exit the current program with error code 0 (success).
  * @note This function does not return.
  */
void exit()
{
    exit(0);
}

/** Exit the current program with an explicit error code.  Most systems allow
  * an error code in the range of 0 to 127; this is neither a minimum nor
  * maximum valid range.
  *
  * @param code The error code to return to the system.
  * @note This function does not return.
  */

void exit(int code)
{
    sysFunctions.exit(code);
}

/** Split a path into head and tail components, where the head contains
  * everything up to the final component and tail is the final component.
  * For example, pathTail("a/b/c", head, tail) puts "a/b" in head and "c" in
  * tail.  If there is no head and/or no tail, there is none given.
  *
  * @param path The path to split.
  * @param head The path before the final component.
  * @param tail The final component of the path.
  */

void pathTail(char[] path, out char[] head, out char[] tail)
{
    for (int c = path.length - 1; c >= 0; c --)
    {
        if (path[c] == '/' || path[c] == '\\')
        {
            head = path[0..c];
            tail = path[c + 1..path.length];
            return;
        }
    }

    head = null;
    tail = path;
}
unittest
{
    char [] head, tail;

    pathTail("a/b", head, tail); assert(head == "a"); assert(tail == "b");
    pathTail("a\\b/c", head, tail); assert(head == "a\\b"); assert(tail == "c");
    pathTail("/a", head, tail); assert(head == ""); assert(tail == "a");
    pathTail("a/", head, tail); assert(head == "a"); assert(tail == "");
    pathTail("a", head, tail); assert(head == ""); assert(tail == "a");
}

/** Returns whether the file or directory exists.
  * @param path The path to the resource to test.
  * @return Returns true if this file or directory exists or null otherwise.
  */

bit exists(char[] path)
{
    sysFunctions.sstat st;
    return sysFunctions.stat(path, &st) == 0;
}

/** Create a directory.
  *
  * @param path The path to create.  This can be relative or absolute
  * and can have sub-paths.  For example, "a/b/c" is correct, "../b" is
  * correct, and "/a/b/c" is correct.  This will create any intermediaries.
  */

void mkdir(char[] path)
{
    char[] head, tail;

    pathTail(path, head, tail);
    if (tail == null)
        pathTail(head, head, tail);
    if (head != null && tail != null && !exists(head))
        mkdir(head);
    sysFunctions.mkdir(toStringz(path));
}

/** Remove a directory.
  *
  * @param path The directory to remove.
  * @param force If this is set, then any content in the directory will be
  * deleted before removing the directory.  Otherwise if you try to delete
  * a directory with files in it, this function will throw an exception.
  */

void rmdir(char[] path, bit force)
{
    void deleteContents(char[] path, Stat[] list)
    {
        for (int c; c < list.length; c ++)
            if (list[c].isdir)
                sysFunctions.rmdir(toStringz(list[c].filename));
            else
                std.file.remove(list[c].filename);
    }

    if (force)
        walk(path, &deleteContents, true);
    sysFunctions.rmdir(toStringz(path));
}

/** Walk through a tree of directories, calling func in each directory.
  */

void walk(char[] path, void delegate(char[] path, Stat[] list) func, bit depthFirst)
{
    void dir(char[] path)
    {
        Stat[] list = listdir(path);

        if (!depthFirst)
            func(path, list);

        for (int c; c < list.length; c ++)
            if (list[c].isdir)
                dir(list[c].filename);

        if (depthFirst)
            func(path, list);
    }

    dir(path);
}

/** Get a list of contents of a directory.  This does not include "." and "..".
  */

Stat[] listdir(char[] path)
{
    Stat[] list;

    path = path.dup;
    if (path.length && (path[path.length - 1] != '/' && path[path.length - 1] != '\\' && path[path.length - 1] != ':'))
        path ~= "/";
    char[] opath = path;
    path ~= "*.*";

    WIN32_FIND_DATA data;
    HANDLE handle;

    handle = FindFirstFileA (path, &data);
    if (handle == (HANDLE) 0)
        return null;

    try
    {
        while (1)
        {
            Stat item;

            item.basename = std.string.toString(data.cFileName);
            if (item.basename == "." || item.basename == "..")
                goto skip;

            item.filename = opath ~ item.basename;
            item.path = opath;
            item.ext = getExt(item.basename);
            item.size = ((ulong) data.nFileSizeHigh << 32) | data.nFileSizeLow;
            item.isdir = (data.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY) != 0;
            list ~= item;

        skip:
            if (!FindNextFileA (handle, &data))
                break;
        }
    }
    finally
        FindClose (handle);

    return list;
}